#! /usr/bin/perl -w
#
#  backups.pl  - 2004.03.03 Yann GROSSEL
#
#  Ce script doit tre plac sur les machines ou se trouvent des fichiers/rpertoires  backuper.
#
#  Ncessite normalement Perl 5.6.1 ou suprieur.
#
#  (Semble toutefois fonctionner plus ou moins avec perl 5.0003...)
#

# Import des modules utiliss

use strict;
use File::Temp qw (tempfile);
use File::Find;

# Dfinitions des variables

my $base  = '/usr/local/Claranet/Configs-Backups';
# my $base  = '/home/yg/Projects/Claranet/Local/Configs-Backups/Test-Client';

my $p_pf  = '/var/tmp/clara-backup.lst';			# Fichier utilis pour stocker les permissions, uids, gids
my $p_pk  = '/var/tmp/cb.packages.lst';			# Fichier utilis pour stocker la liste des packages de la machine.

my $sopts = '-2 -F ' . $base . '/ssh.config';
my @ropts = ("-arHx", "--delete", "--no-implied-dirs");
my $conf  = $base . '/backups.conf';

my $lcd   = '$LastChangedDate: 2005-07-13 21:00:56 +0200 (Wed, 13 Jul 2005) $';
my $lcr   = '$LastChangedRevision: 36 $';

my ($d, $m, $y) =  ((localtime())[3, 4, 5]);
my $suffix = "--suffix=" . sprintf(".%.4d-%.2d-%.2d", $y + 1900, $m + 1, $d);

my ($ssh, $rsync, @targets, @glob_excluded, %files);
my ($h_ex, $p_ex, $h_vr, $p_vr, $h_nv, $p_nv, $n_ex, $n_vr, $n_nv);	# explications noms des vars dans read_config()

# Procdures

sub add_file()
{
   my ($mode, $uid, $gid) = (lstat $File::Find::name)[2, 4, 5];
	if (not defined $mode) { print "Can't lstat ${File::Find::name}: $!\n"; return; }
   $files{$File::Find::name} = sprintf "%o %d %d", $mode & 07777, $uid, $gid;
}

sub read_config()
{
   open F, $conf or die "\nCan't open $conf: $!\n\n";

   ($h_ex, $p_ex) = tempfile(DIR => '/var/tmp');     # Excluded files, handle & path
   ($h_vr, $p_vr) = tempfile(DIR => '/var/tmp');     # Versionned files, handle & path
   ($h_nv, $p_nv) = tempfile(DIR => '/var/tmp');     # Non-Versionned files, handle & path

   ($n_ex, $n_vr, $n_nv) = (0, 0, 0);					  # Number of files (excluded, versionned, non-versionned)

	my $find_options = { 'wanted' => \&add_file, 'no_chdir' => 1 };

   while (<F>) {
      chomp;

      if (m/^\s*rsync\s*=\s*(\S+)/o)      { $rsync = $1; next; }
      if (m/^\s*ssh\s*=\s*(\S+)/o)        { $ssh = $1; next; }
      if (m/^\s*targets\s*=\s*(.*)\s*$/o) { @targets = split m/\s+/o, $1; next; }

      if (m#^\s*!\s*/#o) {									  # Exclusion pattern
			s#^\s*!\s*##o;
			print $h_ex $_ . chr(0);
			$n_ex = 1;
			push @glob_excluded, glob($_);
			next;
		}

      if (m#^\s*=\s*/#o) {									  # Non versionned backup file/dir
			s#^\s*=\s*##o;
			for my $f (glob($_)) {
				find($find_options, $f);
				$f =~ s#^/##o;
				print $h_nv $f . chr(0);
				$n_nv = 1;
			}
			next;
		}

      if (m#^\s*/#o) {										  # Versionned backup file/dir
			s#^\s*##o;
			for my $f (glob($_)) {
				find($find_options, $f);
				$f =~ s#^/##o;
				print $h_vr $f . chr(0);
				$n_vr = 1;
			}
			next;
		}
   }

   close F;
}

sub show_version()
{
   my $lc = 'LastChanged';
   $lcd =~ s/^\$${lc}Date: //; $lcd =~ s/ \+.* \$$//;
   $lcr =~ s/^\$${lc}Revision: //; $lcr =~ s/ \$$//;
   print "\nConfigs-Backups, revision $lcr ($lcd)\n\n";
}

sub checks()
{
   die "\nError: rsync not defined in '$conf' !\n\n"   if (not defined($rsync));
   die "\nError: ssh not defined in '$conf' !\n\n"     if (not defined($ssh));
   die "\nError: targets not defined in '$conf' !\n\n" if (not @targets);

   if (!$n_vr and !$n_nv) {
      print "Nothing to backup !\n\n" if (-t STDOUT);
      exit 0;
   }
}

sub build_perms_file()
{
	# Pour tous les fichiers backups (versionned et non-versionned), on note leurs modes, uids et gids.

   my %ge = map { $_ => 1 } @glob_excluded;  # Liste des fichiers exclus

   for my $file (keys %files) {
      my $f = $file;
      while ($f ne '') {
         if (defined($ge{$f})) { delete $files{$file}; last; }
         $f =~ s#/[^/]*$##;						# regex quivalente  un 'dirname'
      }
   }

   open F, '>' . $p_pf;
   for my $f (sort keys %files) { print F $files{$f} . ' ' . $f . "\n"; }
   close F;

   print $h_nv $p_pf . chr(0);  				   # On backupe le fichier contenant les perms/owners, sans versionning.
	$n_nv = 1;
}

sub build_packages_file()
{
	# On dumpe la liste des packages actuellement prsents sur la machine.

	if (-x '/usr/bin/dpkg' && -x '/usr/bin/apt-get') {
		# La machine semble tre une Debian (ou une redhat avec apt-get)
		system("/usr/bin/dpkg --list > $p_pk");
	} elsif (-x '/usr/sbin/pkg_info') {
		# La machine semble tre un FreeBSD
		system("/usr/sbin/pkg_info > $p_pk");
	}

	print $h_nv $p_pk . chr(0);					# On backupe le fichier contenant les packages, sans versionning.
	$n_nv = 1;
}

sub do_backups()
{
   for my $target (@targets) {   # Pour chacune des targets

      my @args;

      push @args, @ropts;
      push @args, "-v", "--progress" if (-t STDOUT);
      push @args, "-e", $ssh . ' ' . $sopts;
      push @args, '--from0';
      push @args, '--exclude-from=' . $p_ex if ($n_ex);

      if ($n_vr) {                 # Versionned Backups
         my @vargs;
         push @vargs, "-b", "--backup-dir=Old-Versions";
         push @vargs, $suffix;
         push @vargs, @args;
         push @vargs, "--files-from=" . $p_vr, '/', $target;
         print $rsync . ' ' . join(' ', @vargs) . "\n\n" if (-t STDOUT);
         system $rsync, @vargs;
      }

      if ($n_nv) {                # Non-Versionned Backups
         push @args, "--files-from=" . $p_nv, '/', $target;
         print $rsync . ' ' . join(' ', @args) . "\n\n" if (-t STDOUT);
         system $rsync, @args;
      }
   }
}

sub close_files()
{
   close $h_vr;                       			 # Fermeture fichiers temporaires
   close $h_nv;
   close $h_ex;
}

sub cleaning()
{
   unlink $p_ex, $p_vr, $p_nv, $p_pf, $p_pk;  # Suppression fichiers temporaires
}

# Code principal

for (@ARGV)
{
	if (m/^-(v|-version)$/) { show_version(); exit 0; }
}

show_version() if (-t STDOUT);
read_config();
checks();
build_perms_file();
build_packages_file();
close_files();
do_backups();
cleaning();

exit 0;

