#! /usr/bin/perl -w

#
#	mail-stats-server.pl	- Yann GROSSEL 2003.07.01
#
#	Ce script cre un serveur UDP. Ce serveur reoit les statistiques de dlai
#	emails envoys par les serveurs de dlivraison de courrier. Il place les
#	statistiques dans une base de donnes MySQL.
#
#	Les donnes prsentes dans la base MySQL sont rgulirement "compresses" un peu
#	 la faon d'une base de donne RRD, par dplacement vers des tables contenant
#	des donnes de moins en moins prcises.
#

# TODO Supprimer les records complets de la table daily dont l'age est compris entre 5j12h et 33h20
# (les records antrieurs  5j12h tant dj tous supprims)
#
# TODO Stocker le PID dans un fichier lock.
# TODO Permettre une option pour killer le daemon en cours (par examen du fichier de PID)
# TODO Permettre une option pour dmarrer en mode dbug (non fork, debug stdout).

use strict;
use IO::Socket;
use DBI;
use POSIX;

## Configuration #############################################################

my $listen_port = 18820;			# Port UDP sur lequel on doit attendre les messages

my	$db_host = 'localhost';			# Hostname du serveur MySQL  utiliser
my $db_base = 'mail_stats';		# Base MySQL  utiliser
my $db_user = 'mail_stats';		# User MySQL  utiliser
my $db_pass = 'gpRccQ6VE4';		# Password de l'user MySQL

##############################################################################

sub mysql_connect()
{
	# Effectue une connexion (ou une reconnexion) au serveur mysql.
	# Return 1 si OK, sinon 0.

	if (defined($::dbh))
	{
		$::dbh->disconnect();
		undef $::dbh;
	}

	if (! ($::dbh = DBI->connect("dbi:mysql:database=$db_base", $db_user, $db_pass)))
	{
		warn "Can't connect to MySQL !\n";
		return 0;
	}

	return 1;
}

sub mysql_query($)
{
	my $r = shift;
	my $req = $::dbh->prepare($r);
	if (! $req->execute())
	{
		warn "Can't execute MySQL request.\n";
		if (!&mysql_connect()) { return undef; }
		$req = $::dbh->prepare($r);
		if (! $req->execute()) { return undef; }
	}
	return $req;
}

sub daemonize()
{
	# Passage en mode daemon

	# TODO Pour bien faire il faudrait un mcanisme de locking pour empecher plusieurs instances du
	# programme d'tre dmarr en mme temps. C'est vrai qu'une seule des instances pourra binder
	# la socket sur le port d'coute mais bon... c'est plutt moche comme mthode.

	# On fork()e.

	my $pid = fork();
	exit if $pid;
	die "Couldn't fork: $!" unless defined($pid);

	# On dmarre une nouvelle session

	POSIX::setsid() or die "Can't start a new session: $!";
}

sub load_sources()
{
	# Chargement des sources de donnes que l'on doit grer.
	# Cette fonction effectue un cache de 4 minutes des informations pour viter de faire trop de
	# requetes sur la base de donnes.

	my $last if 0;	# (dclaration de variable statique en perl)

	return if (defined($last) and ($last + 240) > time());

	$last = time();

	@::sources = ();

	my $req = &mysql_query("SELECT id, host, target FROM sources");

	return unless defined($req);

	while (my @source = $req->fetchrow_array()) {
		push @::sources, \@source;
	}
}

sub find_source($$)
{
	# Recherche l'id d'une source dans la liste des sources courantes.
	# Renvoie undef si la source demande n'est pas connue.

	my $relay = shift;	# Relai SMTP utilis
	my $target = shift;	# Adresse email cible utilise

	load_sources();		# Charge les sources dfinies actuellement depuis la base de donnes.

	for my $s (@::sources) {
		return @$s[0] if ($relay eq @$s[1] && $target eq @$s[2]);
	}

	return undef;
}

sub tables_slots($$)
{
	# Insre les records ncessaires pour un timestamp et pour une source donne, dans toutes les tables.
	# Si le slot existe dj pour le timestamp et la source indiqus, on ne fait rien.
	# On garde un cache des dernirs slots ajouts, afin de ne pas avoir  rechecker sans arret dans la
	# base MySQL. Ce cache est gr grce au hash %::slots. Les keys sont au format "$timestamp:$id". Les
	# valeurs sont l'heure d'ajout dans le cache. Une fois par heure les valeurs prsentes dans le cache
	# depuis plus d'une heure sont expires.

	my $last if 0;	# (dclaration de variable statique en perl)

	if (defined($last) and ($last + 3600) < time())
	{
		# Expiration des valeurs prsentes depuis plus d'une heure dans le cache.

		print "Expiring slots cache !\n" if $::debug;

		my @to_remove;

		for my $k (keys %::slots) { push @to_remove, $k if ($::slots{$k} < $last); }
		for my $k (@to_remove) { delete $::slots{$k}; }

		$last = time();
	}
	elsif (not defined($last))
	{
		$last = time();
	}

	#

	my $ts = shift;							# Timestamp concern.
	my $source = shift;						# Source concerne.

	my $k = $ts . ':' . $source;			# Cl du slot dans le cache.

	return if (defined($::slots{$k}));	# Le slot est dj dans le cache : on ne fait rien.

	$::slots{$k} = time();					# On ajoute le slot dans le cache.

	# On vrifie si le slot existe dj dans la base; et on l'insre si ncessaire.

	my $req;						# Servira pour les requtes SQL
	my @row;						# Servira pour les rsultats des requtes SQL
	my $index = 0;				#

	for my $table (@::tables)
	{
		$ts -= ($ts % $::rounds[$index++]);		# On arondit le timestamp  la granularit de la table.

		$req = &mysql_query("SELECT COUNT(*) FROM $table WHERE ts = $ts AND source_id = $source");

		return unless defined($req);

		@row = $req->fetchrow_array();

		if ($row[0] == 0)
		{
			# Slot inexistant de record, on doit donc l'insrer.
			$::dbh->do("INSERT INTO $table (id, ts, source_id, value) VALUES (NULL, $ts, $source, NULL)");
		}
	}
}

sub archive_table($$$$)
{
	my $round			= shift;		# Nombre de secondes
	my $table_from		= shift;		# Table source des donnes (table "infrieure")
	my $table_to		= shift;		# Table de destination des donnes (table "suprieure")
	my $n_data			= shift;		# Nombre de donnes de la table source quivalent  1 donne de la table cible

	my $req;				# Contiendra le rsultat de la requte SQL
	my %recs;			# Pour chaque source existante, id des records concerns (rference sur array)
	my %vals;			# Pour chaque source existante, donnes sur une priode (rference sur array)
	my %fvls;			# f_values (nombre de valeurs atomiques utilises dans le calcul)
	my $base_ts = 0;	# Timestamp de base courant.

	# On examine tous les records de  la table $table_from.
	# On travaille sur des groupes de $n_data donnes dans la table source.
	# On met  jour les donnes de la table destination en consquence.

	# On dfinit les instructions d'archivage dans une inner-sub-routine, car ces instructions vont tre appelles plusieurs
	# fois. Cette inner-sub-routine doit partager plein de variables avec la outer-sub-routine (&archive_table). On dfinit
	# donc la inner-sub-routine en anonyme (sub {}) comme indiqu dans "Programming Perl -> Diagnostic Messages -> Variable "%s"
	# will not stay shared".

	my $do_archive_group = sub
	{
		# Cette sous-fonction (anonyme) archive les donnes prsentes en RAM dans les tables qui vont bien.
		# Note: les variables sont partages avec la outer-sub-routine (&archive_table) (ces variables ne sont pas modifies
		# par la fonction de toute manire, elles sont juste utilises).

		my $req;		# Pour ne pas partager nos requtes avec celles de la outer-sub-routine

		for my $s (keys(%recs))									# Pour chacune des sources dfinies
		{
			my $avg;													# Valeur moyenne
			my $rows;

			for my $v (@{$vals{$s}}) { $avg += $v };		# On ajoute les valeurs
			$avg /= scalar(@{$vals{$s}});						# On divise par le nombre de valeurs.

			# On tente une rimplmentation de round()

			$avg = sprintf("%.1f", $avg);

			# On vrifie le nombre de rows correspondantes

			$req = &mysql_query("SELECT value, f_values FROM $table_to WHERE ts = $base_ts AND source_id = $s");

			return unless defined($req);

			$rows = $req->rows();

			if ($rows > 1)
			{
				# Plusieurs rows ?!

				$::dbh->do("DELETE FROM $table_to WHERE ts = $base_ts AND source_id = $s");
				$rows = 0;
			}

			if ($rows == 0)
			{
				# On doit insrer un record dans la table $table_to.
				$::dbh->do("INSERT INTO $table_to (id, ts, source_id, value, f_values) VALUES (NULL, $base_ts, $s, $avg, $fvls{$s})");
			}
			else
			{
				# Une row : on compare les valeurs, voir si on a les memes.

				my @row = $req->fetchrow_array();

				if ($row[0] != $avg || $row[1] != $fvls{$s})
				{
					$::dbh->do("UPDATE $table_to SET value = $avg, f_values = $fvls{$s} WHERE ts = $base_ts AND source_id = $s");
				}
			}
		}
	};			# Le ';' est ncessaire ici (dfinition de fonction anonyme).

	# 	On examine TOUTES les donnes NON NULLES de la table $table_from.

	my $add = ($table_from ne 'daily')? 'f_values' : 1;

	$req = &mysql_query("SELECT id, ts, source_id, value, $add FROM $table_from WHERE value IS NOT NULL ORDER BY ts");

	return unless defined($req);

	while (my @res = $req->fetchrow_array())				# Pour chaque donne, dans l'ordre chronologique
	{
		# $res[0] = id												# ID de l'enregistrement
		# $res[1] = ts												# Timestamp de la donne
		# $res[2] = source_id									# Source de la donne
		# $res[3] = value											# Valeur de la donne
		# $res[4] = f_values										# Nombre de donnes "atomiques" ayant particip au calcul de cette valeur

		my $ts = $res[1] - ($res[1] % $round);				# Timestamp du groupe auquel appartient la donne

		if ($base_ts != $ts)
		{
			# On a chang de groupe. On archive les donnes du groupe prcedent, prsentes en RAM (s'il y en a).

			if ($base_ts) { &$do_archive_group(); }

			# On efface les listes qui taient en RAM.
			%recs = (); %vals = (); %fvls = ();

			# On note le timestamp de base du nouveau groupe.
			$base_ts = $ts;
		}

		# On stocke les donnes en RAM.

		if (not defined $recs{$res[2]})
		{
			# Pas encore de donnes pour cette source -> on alloue les cases correspondantes dans les hashes
			$recs{$res[2]} = [];
			$vals{$res[2]} = [];
			$fvls{$res[2]} = 0;
		}

		push @{$recs{$res[2]}}, $res[0];		# id du record
		push @{$vals{$res[2]}}, $res[3];		# valeur du record

		$fvls{$res[2]} += $res[4];				# On ajoute le nombre de valeurs atomiques
	}

	# On archive les donnes restant en RAM mais non encore traites.
	&$do_archive_group();
}

sub expire_table($$)
{
	my $table = shift;		# Nom de la table  expirer
	my $timeout = shift;		# Nombre de secondes aprs lequel les donnes peuvent etre supprimes de la table.

	# On calcule le timestamp maximum prsent dans la table.

	my $req = &mysql_query("SELECT MAX(ts) FROM $table");

	return unless defined($req);

	my @res = $req->fetchrow_array();

	# On supprime les valeurs

	my $ts = $res[0] - $timeout;

	$::dbh->do("DELETE FROM $table WHERE ts < $ts");
}

sub cleanup_tables()
{
	# Cette fonction examine l'intgralit des tables  la recherche d'erreur. Normalement, aucune erreur ne s'introduit
	# jamais dans les tables ;) Mais il vaut mieux tre un peu prudent, a n'est pas trs difficile.
	#
	# Cette fonction est destine  tre appele au dmarrage du programme puis  intervalles plus ou moins rguliers
	# (toutes les heures par exemple est une bonne ide).

	#	Description de l'algo que l'on va suivre :
	#
	#	- Pour toutes les donnes de la table daily, on les regroupe par groupe de 30 minutes, et on vrifie que le record
	#	  correspondant existe bien dans la table weekly, et que la valeur de ce record correspond bien  celle que l'on
	#	  peut calculer. Si ce n'est pas le cas, on met  jour la table weekly (en insrant un record si ncessaire).
	#	  En passant, on supprime les groupes de records de la table daily qui sont antrieurs  33h20 (puisqu'ils ne seront
	#	  plus graphs). On ne supprime que les groupes COMPLETS - car on est a peu prs certains qu'on ne recevra plus jamais
	#	  de mise  jour pour les donnes de ces groupes. Les groupes incomplets, par contre, restent dans la table daily :
	#	  il est possible qu'un mail soit bloqu dans une queue et mette plus de 33h20  nous parvenir... ce qui modifiera les
	#	  donnes des tables suprieures, mme si cela ne se vera pas sur le graphe daily.
	#
	#	- On fait exactement la mme chose pour les tables weekly -> monthly, et pour les tables monthly -> yearly, sauf que
	#	  pour ces tables l, le probleme des records incomplets ne se pose plus. En effet, les records doivent toujours rester
	#	  prsents dans les tables plus longtemps que le temps maximum de completion (5 jours), afin de pouvoir tre graphs.
	#	  Cela simplifie de beaucoup le probleme : pour ces tables, on se contente de vrifier que les records correspondants
	#	  existent bien et sont  jour dans les tables suprieures - RIEN DE PLUS !
	#
	#	- On supprime les records de la table daily qui sont antrieurs  5 jours. Ca n'est pas la peine de recalculer les
	#	  donnes des tables suprieures en consquence : on est srs que les moyennes ont dj t faites correctement, et
	#	  qu'elles ne vont pas changer (elles ne changent que dans le cas de l'entre d'une nouvelle valeur dans la table
	#	  daily, jamais dans un autre cas).
	#
	#	- De mme, on supprime les records de la table weekly qui sont antrieurs  10 jours (environ -  dterminer).
	#
	#	- De mme, on supprime les records de la table monthly qui sont antrieurs  40 jours (environ -  dterminer).
	#
	#	- De mme, on supprime les records de la table yearly qui sont antrieurs  400 jours (environ -  dterminer).

	#	weekly 	= 30 minutes =  1800 secondes	:  6 x daily
	#	monthly	=  2 heures  =  7200 secondes :  4 x weekly
	#	yearly	= 24 heures  = 86400 secondes : 12 x monthly

	print "CLEANING TABLES !!\n" if $::debug;

	archive_table( 1800,   'daily',  'weekly',  6);
	archive_table( 7200,  'weekly', 'monthly',  4);
	archive_table(86400, 'monthly',  'yearly', 12);

	expire_table('daily', 475200);				# 5j et 12h
	expire_table('weekly', 864000);				# 10j
	expire_table('monthly', 3024000);			# 35j
	expire_table('yearly', 35424000);			# 410j
}

sub got_value($$$)
{
	# Prend en compte une valeur pour une source et un timestamp.
	# La valeur est d'abord intgre dans la table daily, mais le changement induit est repercut immdiatement
	# dans les tables suprieures,  savoir weekly, monthly et yearly, en faisant  chaque fois le calcul moyenne
	# avec les autres valeurs disponibles pour le groupe.
	#
	# Note 1: pour aller vraiment rapidement, et aussi par mesure de simplification, cette fonction se contente de faire des
	# updates sur les donnes. En aucun cas, on ne va supprimer des donnes, mme si certaines donnes deviennent inutiles
	# car trop anciennes ou car intgres dans une table suprieure. C'est la fonction cleanup_tables() est charge de
	# rgulirement faire du mnage dans les tables.
	#
	# Note 2: il faut pour que cela fonctionne correctement que les records soient dj prsents dans les tables,
	# puisqu'on ne fait que des UPDATE. On s'en assure en appellant la fonction tables_slots().

	my ($ts, $source, $value) = @_;

	tables_slots($ts, $source);					# Cration des slots si ncessaire.

	# On met d'abord  jour la table daily.

	$ts -= ($ts % 300);	# On impose la granularit  5 minutes.

	$::dbh->do("UPDATE daily SET value = $value WHERE ts = $ts AND source_id = $source");

	# On rpercute le changement dans les 3 autres tables, l'une aprs l'autre.

	for my $index (1, 2, 3) {

		my ($req, @row, $avg);

		$ts -= ($ts % $::rounds[$index]);	# On impose la granularit.

		my $field = ($index == 1)? '1' : 'f_values';

		$req = &mysql_query("SELECT SUM(value), COUNT(*), SUM($field) FROM " . $::tables[$index - 1] . " WHERE ts BETWEEN $ts AND " . ($ts + $::rounds[$index] - 1) . " AND source_id = $source AND value IS NOT NULL");

		return unless defined($req);

		@row = $req->fetchrow_array();

		# Visiblement il peut survenir que la requete renvoie des valeurs NULL ou  0 (mais pourquoi ?! telle est
		# la question...). Donc, prcaution : on teste si ce n'est pas le cas, pour viter les problmes
		# de division par 0 ainsi que l'arret subit du programme...

		if (defined($row[0]) && defined($row[1]) && $row[1] > 0)
		{
			$avg = $row[0] / $row[1];

			$avg = sprintf("%.1f", $avg);

			$::dbh->do("UPDATE " . $::tables[$index] . " SET value = $avg, f_values = " . $row[2] . " WHERE ts = $ts AND source_id = $source");
		}
	}

	# Toutes les 2 heures, on cleanup les tables.

	my $last if 0;	# (dclaration de variable statique en perl)

	return if (defined($last) and ($last + 7200) > time());		# 7200 secondes == 2 h

	$last = time();

	cleanup_tables();
}

sub main()
{
	my $server;
	my $peer;
	my $datagram;

	@::tables = ('daily', 'weekly', 'monthly', 'yearly');		# Nom des tables
	@::rounds = (    300,     1800,      7200,    86400);		# Granularit des timestamp pour chaque table

	daemonize() unless $::debug;

	&mysql_connect();

	@::sources = ();
	%::slots = ();

	$server = IO::Socket::INET->new(LocalPort => $listen_port, Proto => "udp")
		or die "Can't start udp server on port $listen_port : $@\n";

	cleanup_tables() unless $::debug;			# On effectue un premier cleanup des tables.

	print "Ready (debug mode)\n" if $::debug;

	while ($peer = $server->recv($datagram, 1024)) {

		chomp $datagram;

		print "DATAGRAM : '$datagram'\n" if $::debug;

		if ($datagram =~ m/^MAIL-STATS-M (\d+) ([\w.\@-]+) ([\w.-]+)/)
		{
			# Un message confirmant qu'un email a bien t envoy.

			my ($when, $target, $relay) = ($1, $2, $3);

			print "Relay = '$relay', Target = '$target'\n" if $::debug;

			my $id = find_source($relay, $target);	# Recherche l'id de la source

			if ($::debug)
			{
				if (defined($id)) { print "-> source_id = $id\n"; }
				else { print "-> UNKNOWN SOURCE !\n"; }
			}

			next if (not defined($id));				# Si source inconnue, on ignore.

			tables_slots($when, $id);					# Cration des slots si ncessaire.
		}
		elsif ($datagram =~ m/^MAIL-STATS-R (\d+) ([\w.\@-]+) ([\w.-]+) -?([\d.]+) -?([\d.]+)/)
		{
			# Un message indiquant la reception d'un des emails de stats.

			my ($when, $target, $relay, $delay, $greet_delay) = ($1, $2, $3, $4, $5);

			print "Relay = '$relay' Target = '$target'\n" if $::debug;

			my $id = find_source($relay, $target);	# Recherche l'id de la source

			if ($::debug)
			{
				if (defined($id)) { print "-> source_id = $id\n"; }
				else { print "-> UNKNOWN SOURCE !\n"; }
			}

			next if (not defined($id));				# Si source inconnue, on ignore.

			got_value($when, $id, $delay);
		}
		else
		{
			if ($::debug)
			{
				print "Unknown datagram format !\n";
			}
		}
	}

	# Dans quel cas on peut sortir de cette boucle ? C'est  dire, dans quel cas le recv() peut-il
	# renvoyer une erreur...

	exit 0;
}

$::debug = (scalar(grep(/^-d$/, @ARGV)))? 1 : 0;

&main();

##############################################################################
#
#	EXPLICATIONS
#	============
#
#	Bases de donnes
#
#	1 table pour les sources de donnes :
#
#	- sources
#
#	(pour le moment cette table contiendra 8 records)
#
#	TODO Amliorer cette table pour contenir des positions (ou placer les graphes correspondants sur la page gnrale)
#	et d'autres trucs du mme genre...
#
#	4 tables pour les donnes :
#
#	- daily			une valeur toutes les 5 minutes
#	- weekly			une valeur toutes les 30 minutes		(6 * 5  minutes)
#	- monthly		une valeur toutes les 2 heures		(4 * 30 minutes)
#	- yearly			une valeur tous les jours				(12 * 2 heures)
#
#	* Intgration des donnes arrivant : dans la table daily
#
#	La table daily contient les donnes du jour et aussi toutes les autres donnes tant qu'elles n'ont
#	pas t intgres dans les autres tables pour cause de non compltion.
#
#	Lors de la rception d'un STATS-S, on ajoute dans la table daily autant de records qu'il existe de sources,
#	avec comme valeur NULL.
#
#	Ensuite lorsque les STATS-R arrivent, on met  jour les valeurs correspondantes dans la table daily, et on
#	met aussi  jours les valeurs correspondantes dans les 3 autres tables (en recalculant les moyennes en
#	fonction des autres valeurs existantes).
#
#	Si plusieurs STATS-R arrivent pour une mme source et un mme timestamp, on remplace aveuglement l'ancienne
#	valeur.
#
#	Si un STATS-R arrive pour une source inconnue ou un timestamp sans record correspondant, on l'ignore royalement.
#
#	TODO TIMEOUT DES DONNEES TOUJOURS SANS VALEUR AU BOUT DE 5 JOURS
#
######################################
#
#
